﻿using Newtonsoft.Json.Linq;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Net.Http;
using System.Threading;
using System.Threading.Tasks;

namespace WikipediaRagWinForms.Services
{
    public class WikipediaOnlineClient
    {
        private readonly HttpClient _http;
        private static readonly Dictionary<string, string> _summaryCache =
    new Dictionary<string, string>(StringComparer.OrdinalIgnoreCase);

        public WikipediaOnlineClient()
        {
            _http = new HttpClient();
            _http.DefaultRequestHeaders.Add(
                "User-Agent",
                "Mozilla/5.0 (Windows NT 10.0; Win64; x64)"
            );
        }

        // Nutzt den übergebenen Suchbegriff 1:1 in der Wikipedia-Suche.
        public async Task<string> SearchAsync(string query)
        {
            // 1. Sicherheitskonvertierung
            if (string.IsNullOrWhiteSpace(query))
                return "";

            string url =
                "https://de.wikipedia.org/w/api.php?action=query&list=search&srinfo=suggestion&format=json&utf8=1&srsearch="
                + Uri.EscapeDataString(query);

            try
            {
                using (var client = new HttpClient())
                {
                    string json = await client.GetStringAsync(url);
                    var root = Newtonsoft.Json.Linq.JObject.Parse(json);

                    var first = root["query"]?["search"]?.First;
                    if (first != null)
                    {
                        string title = (string)first["title"];
                        if (!string.IsNullOrWhiteSpace(title))
                            return title;
                    }

                    // 2. Wikipedia-Vorschläge ("did you mean")
                    string suggestion = (string)root["query"]?["searchinfo"]?["suggestion"];
                    if (!string.IsNullOrWhiteSpace(suggestion))
                        return suggestion;
                }
            }
            catch
            {
                // Ignorieren – wir bauen Fallbacks ein
            }

            // 3. KI behandelt "Magnus Carlsen"-Fragen → immer sinnvoll
            if (query.IndexOf("Schach", StringComparison.OrdinalIgnoreCase) >= 0)
                return "Schachweltmeister";

            // 4. Letzter Fallback: gib die Eingabe zurück
            return query;
        }


        // Kurz-Zusammenfassung (Summary) laden // Cache Version
        public async Task<string> GetSummaryAsync(string title)
        {
            if (_summaryCache.TryGetValue(title, out var cached))
                return cached;

            string url = "https://de.wikipedia.org/api/rest_v1/page/summary/" +
                         Uri.EscapeDataString(title);

            try
            {
                // selben HttpClient verwenden → User-Agent bleibt gesetzt
                string json = await _http.GetStringAsync(url);
                var obj = Newtonsoft.Json.Linq.JObject.Parse(json);

                string extract = (string)obj["extract"] ?? "";

                _summaryCache[title] = extract;

                return extract;
            }
            catch
            {
                return "";
            }

        }
                
        public async Task<string> GetFullExtractAsync(string title)
        {
            string url =
                "https://de.wikipedia.org/w/api.php?action=query&prop=extracts&explaintext=1&format=json&redirects=1&titles="
                + Uri.EscapeDataString(title);

            try
            {
                string json = await _http.GetStringAsync(url);
                var root = JObject.Parse(json);

                var pages = root["query"]?["pages"];
                if (pages != null)
                {
                    foreach (var p in pages.Children())
                    {
                        string extract = (string)p.First["extract"];
                        if (!string.IsNullOrWhiteSpace(extract))
                            return extract;
                    }
                }
            }
            catch
            {
            }

            return "";
        }
    }
}
